import getAllCookies from './modules/get_all_cookies.mjs';
import { jsonToNetscapeMapper, formatMap } from './modules/cookie_format.mjs';

/** Promise to get URL of Active Tab */
const getUrlPromise = chrome.tabs.query({ active: true, currentWindow: true }).then(([{ url }]) => new URL(url));

// ----------------------------------------------
// Functions
// ----------------------------------------------

/**
 * Get Stringified Cookies Text and Format Data
 * @param {chrome.cookies.GetAllDetails} details
 * @returns {Promise<{ text: string, format: Format }>}
 */
const getCookieText = async (details) => {
  const cookies = await getAllCookies(details);
  const format = formatMap['netscape'];
  if (!format) throw new Error('Invalid format');
  const text = format.serializer(cookies);
  return { text, format };
};

/**
 * Save text data as a file
 * Firefox fails if revoked during download.
 * @param {string} text
 * @param {string} name
 * @param {Format} format
 * @param {boolean} saveAs
 */
const saveToFile = async (text, name, { ext, mimeType }, saveAs = false) => {
  const blob = new Blob([text], { type: mimeType });
  const filename = name + ext;
  const url = URL.createObjectURL(blob);
  const id = await chrome.downloads.download({ url, filename, saveAs });

  /** @param {chrome.downloads.DownloadDelta} delta  */
  const onChange = (delta) => {
    if (delta.id === id && delta.state?.current !== 'in_progress') {
      chrome.downloads.onChanged.removeListener(onChange);
      URL.revokeObjectURL(url);
    }
  };

  chrome.downloads.onChanged.addListener(onChange);
};

/**
 * Copy text data to the clipboard
 * @param {string} text
 */
const setClipboard = async (text) => {
  await navigator.clipboard.writeText(text);
  //document.getElementById('copy').innerText = 'Copied!';
};

// ----------------------------------------------
// Actions after resolving the promise
// ----------------------------------------------

/** Set URL in the header */
/**
getUrlPromise.then((url) => {
  const location = document.querySelector('#location');
  location.textContent = location.href = url.href;
});
*/

/** Set Cookies data to the table */
/**
getUrlPromise
  .then((url) => getAllCookies({ url: url.href, partitionKey: { topLevelSite: url.origin } }))
  .then((cookies) => {
    const netscape = jsonToNetscapeMapper(cookies);
    const tableRows = netscape.map((row) => {
      const tr = document.createElement('tr');
      tr.replaceChildren(
        ...row.map((v) => {
          const td = document.createElement('td');
          td.textContent = v;
          return td;
        })
      );
      return tr;
    });
    document.querySelector('table tbody').replaceChildren(...tableRows);
  });
*/


// ----------------------------------------------
// Event Listeners
// ----------------------------------------------

/**
document.querySelector('#export').addEventListener('click', async () => {
  const url = await getUrlPromise;
  const details = { url: url.href, partitionKey: { topLevelSite: url.origin } };
  const { text, format } = await getCookieText(details);
  saveToFile(text, url.hostname + '_cookies', format);
});

document.querySelector('#exportAs').addEventListener('click', async () => {
  const url = await getUrlPromise;
  const details = { url: url.href, partitionKey: { topLevelSite: url.origin } };
  const { text, format } = await getCookieText(details);
  saveToFile(text, url.hostname + '_cookies', format, true);
});

document.querySelector('#copy').addEventListener('click', async () => {
  const url = await getUrlPromise;
  const details = { url: url.href, partitionKey: { topLevelSite: url.origin } };
  const { text } = await getCookieText(details);
  setClipboard(text);
});

document.querySelector('#exportAll').addEventListener('click', async () => {
  const { text, format } = await getCookieText({ partitionKey: {} });
  saveToFile(text, 'cookies', format);
});
*/


document.querySelector('#dllink').addEventListener('click', async () => {
  { //
    const details = await chrome.runtime.sendMessage({type: 'GET_MATCHED_REQUESTS'});
    const urls = details.requests.map(r => r.url);
    urls.forEach(url => {
      //console.log('Test url:', url);
      setClipboard(url);
    });
  }
  {
    const details = await chrome.runtime.sendMessage({type: 'GET_POTOKEN_REQUESTS'});
    const poTokens = details.requests.map(r => r.data);
    poTokens.forEach(poToken => {
      //console.log('poToken: ', url);
      setClipboard('poToken: '+poToken);
    });
  }
  {
    const prefix='#aigosoft-url:';
    const url = await getUrlPromise;
    const details = { url: url.href, partitionKey: { topLevelSite: url.origin } };
    const { text } = await getCookieText(details);
    setClipboard(prefix+url+'\n'+text);
  }
});

document.querySelector('#exportAs').addEventListener('click', async () => {
  const url = await getUrlPromise;
  const details = { url: url.href, partitionKey: { topLevelSite: url.origin } };
  const { text, format } = await getCookieText(details);
  saveToFile(text, url.hostname + '_cookies', format, true);
});
/*
document.querySelector('#exportAll').addEventListener('click', async () => {
  const { text, format } = await getCookieText({ partitionKey: {} });
  saveToFile(text, 'cookies', format);
});
*/
document.querySelector('#homeLink').addEventListener('click', async () => {
  chrome.tabs.create({ url: 'https://www.videoxget.com/res/how-to-use-getcookies.html' });
});
